// SPDX-License-Identifier: GPL-2.0-only
/*
 * Copyright (C) 2011 LAPIS Semiconductor Co., Ltd.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/string.h>
#include <linux/pci.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/mutex.h>
#include <linux/if_ether.h>
#include <linux/ctype.h>
#include <linux/dmi.h>
#include <linux/of.h>

#define PHUB_STATUS 0x00		/* Status Register offset */
#define PHUB_CONTROL 0x04		/* Control Register offset */
#define PHUB_TIMEOUT 0x05		/* Time out value for Status Register */
#define PCH_PHUB_ROM_WRITE_ENABLE 0x01	/* Enabling for writing ROM */
#define PCH_PHUB_ROM_WRITE_DISABLE 0x00	/* Disabling for writing ROM */
#define PCH_PHUB_MAC_START_ADDR_EG20T 0x14  /* MAC data area start address
					       offset */
#define PCH_PHUB_MAC_START_ADDR_ML7223 0x20C  /* MAC data area start address
						 offset */
#define PCH_PHUB_ROM_START_ADDR_EG20T 0x80 /* ROM data area start address offset
					      (Intel EG20T PCH)*/
#define PCH_PHUB_ROM_START_ADDR_ML7213 0x400 /* ROM data area start address
						offset(LAPIS Semicon ML7213)
					      */
#define PCH_PHUB_ROM_START_ADDR_ML7223 0x400 /* ROM data area start address
						offset(LAPIS Semicon ML7223)
					      */

/* MAX number of INT_REDUCE_CONTROL registers */
#define MAX_NUM_INT_REDUCE_CONTROL_REG 128
#define PCI_DEVICE_ID_PCH1_PHUB 0x8801
#define PCH_MINOR_NOS 1
#define CLKCFG_CAN_50MHZ 0x12000000
#define CLKCFG_CANCLK_MASK 0xFF000000
#define CLKCFG_UART_MASK			0xFFFFFF

/* CM-iTC */
#define CLKCFG_UART_48MHZ			(1 << 16)
#define CLKCFG_UART_25MHZ			(2 << 16)
#define CLKCFG_BAUDDIV				(2 << 20)
#define CLKCFG_PLL2VCO				(8 << 9)
#define CLKCFG_UARTCLKSEL			(1 << 18)

/* Macros for ML7213 */
#define PCI_DEVICE_ID_ROHM_ML7213_PHUB		0x801A

/* Macros for ML7223 */
#define PCI_DEVICE_ID_ROHM_ML7223_mPHUB	0x8012 /* for Bus-m */
#define PCI_DEVICE_ID_ROHM_ML7223_nPHUB	0x8002 /* for Bus-n */

/* Macros for ML7831 */
#define PCI_DEVICE_ID_ROHM_ML7831_PHUB 0x8801

/* SROM ACCESS Macro */
#define PCH_WORD_ADDR_MASK (~((1 << 2) - 1))

/* Registers address offset */
#define PCH_PHUB_ID_REG				0x0000
#define PCH_PHUB_QUEUE_PRI_VAL_REG		0x0004
#define PCH_PHUB_RC_QUEUE_MAXSIZE_REG		0x0008
#define PCH_PHUB_BRI_QUEUE_MAXSIZE_REG		0x000C
#define PCH_PHUB_COMP_RESP_TIMEOUT_REG		0x0010
#define PCH_PHUB_BUS_SLAVE_CONTROL_REG		0x0014
#define PCH_PHUB_DEADLOCK_AVOID_TYPE_REG	0x0018
#define PCH_PHUB_INTPIN_REG_WPERMIT_REG0	0x0020
#define PCH_PHUB_INTPIN_REG_WPERMIT_REG1	0x0024
#define PCH_PHUB_INTPIN_REG_WPERMIT_REG2	0x0028
#define PCH_PHUB_INTPIN_REG_WPERMIT_REG3	0x002C
#define PCH_PHUB_INT_REDUCE_CONTROL_REG_BASE	0x0040
#define CLKCFG_REG_OFFSET			0x500
#define FUNCSEL_REG_OFFSET			0x508

#define PCH_PHUB_OROM_SIZE 15360

/**
 * struct pch_phub_reg - PHUB register structure
 * @phub_id_reg:			PHUB_ID register val
 * @q_pri_val_reg:			QUEUE_PRI_VAL register val
 * @rc_q_maxsize_reg:			RC_QUEUE_MAXSIZE register val
 * @bri_q_maxsize_reg:			BRI_QUEUE_MAXSIZE register val
 * @comp_resp_timeout_reg:		COMP_RESP_TIMEOUT register val
 * @bus_slave_control_reg:		BUS_SLAVE_CONTROL_REG register val
 * @deadlock_avoid_type_reg:		DEADLOCK_AVOID_TYPE register val
 * @intpin_reg_wpermit_reg0:		INTPIN_REG_WPERMIT register 0 val
 * @intpin_reg_wpermit_reg1:		INTPIN_REG_WPERMIT register 1 val
 * @intpin_reg_wpermit_reg2:		INTPIN_REG_WPERMIT register 2 val
 * @intpin_reg_wpermit_reg3:		INTPIN_REG_WPERMIT register 3 val
 * @int_reduce_control_reg:		INT_REDUCE_CONTROL registers val
 * @clkcfg_reg:				CLK CFG register val
 * @funcsel_reg:			Function select register value
 * @pch_phub_base_address:		Register base address
 * @pch_phub_extrom_base_address:	external rom base address
 * @pch_mac_start_address:		MAC address area start address
 * @pch_opt_rom_start_address:		Option ROM start address
 * @ioh_type:				Save IOH type
 * @pdev:				pointer to pci device struct
 */
struct pch_phub_reg {
	u32 phub_id_reg;
	u32 q_pri_val_reg;
	u32 rc_q_maxsize_reg;
	u32 bri_q_maxsize_reg;
	u32 comp_resp_timeout_reg;
	u32 bus_slave_control_reg;
	u32 deadlock_avoid_type_reg;
	u32 intpin_reg_wpermit_reg0;
	u32 intpin_reg_wpermit_reg1;
	u32 intpin_reg_wpermit_reg2;
	u32 intpin_reg_wpermit_reg3;
	u32 int_reduce_control_reg[MAX_NUM_INT_REDUCE_CONTROL_REG];
	u32 clkcfg_reg;
	u32 funcsel_reg;
	void __iomem *pch_phub_base_address;
	void __iomem *pch_phub_extrom_base_address;
	u32 pch_mac_start_address;
	u32 pch_opt_rom_start_address;
	int ioh_type;
	struct pci_dev *pdev;
};

/* SROM SPEC for MAC address assignment offset */
static const int pch_phub_mac_offset[ETH_ALEN] = {0x3, 0x2, 0x1, 0x0, 0xb, 0xa};

static DEFINE_MUTEX(pch_phub_mutex);

/**
 * pch_phub_read_modify_write_reg() - Reading modifying and writing register
 * @chip:		Pointer to the PHUB register structure
 * @reg_addr_offset:	Register offset address value.
 * @data:		Writing value.
 * @mask:		Mask value.
 */
static void pch_phub_read_modify_write_reg(struct pch_phub_reg *chip,
					   unsigned int reg_addr_offset,
					   unsigned int data, unsigned int mask)
{
	void __iomem *reg_addr = chip->pch_phub_base_address + reg_addr_offset;
	iowrite32(((ioread32(reg_addr) & ~mask)) | data, reg_addr);
}

/* pch_phub_save_reg_conf - saves register configuration */
static void __maybe_unused pch_phub_save_reg_conf(struct pci_dev *pdev)
{
	unsigned int i;
	struct pch_phub_reg *chip = pci_get_drvdata(pdev);

	void __iomem *p = chip->pch_phub_base_address;

	chip->phub_id_reg = ioread32(p + PCH_PHUB_ID_REG);
	chip->q_pri_val_reg = ioread32(p + PCH_PHUB_QUEUE_PRI_VAL_REG);
	chip->rc_q_maxsize_reg = ioread32(p + PCH_PHUB_RC_QUEUE_MAXSIZE_REG);
	chip->bri_q_maxsize_reg = ioread32(p + PCH_PHUB_BRI_QUEUE_MAXSIZE_REG);
	chip->comp_resp_timeout_reg =
				ioread32(p + PCH_PHUB_COMP_RESP_TIMEOUT_REG);
	chip->bus_slave_control_reg =
				ioread32(p + PCH_PHUB_BUS_SLAVE_CONTROL_REG);
	chip->deadlock_avoid_type_reg =
				ioread32(p + PCH_PHUB_DEADLOCK_AVOID_TYPE_REG);
	chip->intpin_reg_wpermit_reg0 =
				ioread32(p + PCH_PHUB_INTPIN_REG_WPERMIT_REG0);
	chip->intpin_reg_wpermit_reg1 =
				ioread32(p + PCH_PHUB_INTPIN_REG_WPERMIT_REG1);
	chip->intpin_reg_wpermit_reg2 =
				ioread32(p + PCH_PHUB_INTPIN_REG_WPERMIT_REG2);
	chip->intpin_reg_wpermit_reg3 =
				ioread32(p + PCH_PHUB_INTPIN_REG_WPERMIT_REG3);
	dev_dbg(&pdev->dev, "%s : "
		"chip->phub_id_reg=%x, "
		"chip->q_pri_val_reg=%x, "
		"chip->rc_q_maxsize_reg=%x, "
		"chip->bri_q_maxsize_reg=%x, "
		"chip->comp_resp_timeout_reg=%x, "
		"chip->bus_slave_control_reg=%x, "
		"chip->deadlock_avoid_type_reg=%x, "
		"chip->intpin_reg_wpermit_reg0=%x, "
		"chip->intpin_reg_wpermit_reg1=%x, "
		"chip->intpin_reg_wpermit_reg2=%x, "
		"chip->intpin_reg_wpermit_reg3=%x\n", __func__,
		chip->phub_id_reg,
		chip->q_pri_val_reg,
		chip->rc_q_maxsize_reg,
		chip->bri_q_maxsize_reg,
		chip->comp_resp_timeout_reg,
		chip->bus_slave_control_reg,
		chip->deadlock_avoid_type_reg,
		chip->intpin_reg_wpermit_reg0,
		chip->intpin_reg_wpermit_reg1,
		chip->intpin_reg_wpermit_reg2,
		chip->intpin_reg_wpermit_reg3);
	for (i = 0; i < MAX_NUM_INT_REDUCE_CONTROL_REG; i++) {
		chip->int_reduce_control_reg[i] =
		    ioread32(p + PCH_PHUB_INT_REDUCE_CONTROL_REG_BASE + 4 * i);
		dev_dbg(&pdev->dev, "%s : "
			"chip->int_reduce_control_reg[%d]=%x\n",
			__func__, i, chip->int_reduce_control_reg[i]);
	}
	chip->clkcfg_reg = ioread32(p + CLKCFG_REG_OFFSET);
	if ((chip->ioh_type == 2) || (chip->ioh_type == 4))
		chip->funcsel_reg = ioread32(p + FUNCSEL_REG_OFFSET);
}

/* pch_phub_restore_reg_conf - restore register configuration */
static void __maybe_unused pch_phub_restore_reg_conf(struct pci_dev *pdev)
{
	unsigned int i;
	struct pch_phub_reg *chip = pci_get_drvdata(pdev);
	void __iomem *p;
	p = chip->pch_phub_base_address;

	iowrite32(chip->phub_id_reg, p + PCH_PHUB_ID_REG);
	iowrite32(chip->q_pri_val_reg, p + PCH_PHUB_QUEUE_PRI_VAL_REG);
	iowrite32(chip->rc_q_maxsize_reg, p + PCH_PHUB_RC_QUEUE_MAXSIZE_REG);
	iowrite32(chip->bri_q_maxsize_reg, p + PCH_PHUB_BRI_QUEUE_MAXSIZE_REG);
	iowrite32(chip->comp_resp_timeout_reg,
					p + PCH_PHUB_COMP_RESP_TIMEOUT_REG);
	iowrite32(chip->bus_slave_control_reg,
					p + PCH_PHUB_BUS_SLAVE_CONTROL_REG);
	iowrite32(chip->deadlock_avoid_type_reg,
					p + PCH_PHUB_DEADLOCK_AVOID_TYPE_REG);
	iowrite32(chip->intpin_reg_wpermit_reg0,
					p + PCH_PHUB_INTPIN_REG_WPERMIT_REG0);
	iowrite32(chip->intpin_reg_wpermit_reg1,
					p + PCH_PHUB_INTPIN_REG_WPERMIT_REG1);
	iowrite32(chip->intpin_reg_wpermit_reg2,
					p + PCH_PHUB_INTPIN_REG_WPERMIT_REG2);
	iowrite32(chip->intpin_reg_wpermit_reg3,
					p + PCH_PHUB_INTPIN_REG_WPERMIT_REG3);
	dev_dbg(&pdev->dev, "%s : "
		"chip->phub_id_reg=%x, "
		"chip->q_pri_val_reg=%x, "
		"chip->rc_q_maxsize_reg=%x, "
		"chip->bri_q_maxsize_reg=%x, "
		"chip->comp_resp_timeout_reg=%x, "
		"chip->bus_slave_control_reg=%x, "
		"chip->deadlock_avoid_type_reg=%x, "
		"chip->intpin_reg_wpermit_reg0=%x, "
		"chip->intpin_reg_wpermit_reg1=%x, "
		"chip->intpin_reg_wpermit_reg2=%x, "
		"chip->intpin_reg_wpermit_reg3=%x\n", __func__,
		chip->phub_id_reg,
		chip->q_pri_val_reg,
		chip->rc_q_maxsize_reg,
		chip->bri_q_maxsize_reg,
		chip->comp_resp_timeout_reg,
		chip->bus_slave_control_reg,
		chip->deadlock_avoid_type_reg,
		chip->intpin_reg_wpermit_reg0,
		chip->intpin_reg_wpermit_reg1,
		chip->intpin_reg_wpermit_reg2,
		chip->intpin_reg_wpermit_reg3);
	for (i = 0; i < MAX_NUM_INT_REDUCE_CONTROL_REG; i++) {
		iowrite32(chip->int_reduce_control_reg[i],
			p + PCH_PHUB_INT_REDUCE_CONTROL_REG_BASE + 4 * i);
		dev_dbg(&pdev->dev, "%s : "
			"chip->int_reduce_control_reg[%d]=%x\n",
			__func__, i, chip->int_reduce_control_reg[i]);
	}

	iowrite32(chip->clkcfg_reg, p + CLKCFG_REG_OFFSET);
	if ((chip->ioh_type == 2) || (chip->ioh_type == 4))
		iowrite32(chip->funcsel_reg, p + FUNCSEL_REG_OFFSET);
}

/**
 * pch_phub_read_serial_rom() - Reading Serial ROM
 * @chip:		Pointer to the PHUB register structure
 * @offset_address:	Serial ROM offset address to read.
 * @data:		Read buffer for specified Serial ROM value.
 */
static void pch_phub_read_serial_rom(struct pch_phub_reg *chip,
				     unsigned int offset_address, u8 *data)
{
	void __iomem *mem_addr = chip->pch_phub_extrom_base_address +
								offset_address;

	*data = ioread8(mem_addr);
}

/**
 * pch_phub_write_serial_rom() - Writing Serial ROM
 * @chip:		Pointer to the PHUB register structure
 * @offset_address:	Serial ROM offset address.
 * @data:		Serial ROM value to write.
 */
static int pch_phub_write_serial_rom(struct pch_phub_reg *chip,
				     unsigned int offset_address, u8 data)
{
	void __iomem *mem_addr = chip->pch_phub_extrom_base_address +
					(offset_address & PCH_WORD_ADDR_MASK);
	int i;
	unsigned int word_data;
	unsigned int pos;
	unsigned int mask;
	pos = (offset_address % 4) * 8;
	mask = ~(0xFF << pos);

	iowrite32(PCH_PHUB_ROM_WRITE_ENABLE,
			chip->pch_phub_extrom_base_address + PHUB_CONTROL);

	word_data = ioread32(mem_addr);
	iowrite32((word_data & mask) | (u32)data << pos, mem_addr);

	i = 0;
	while (ioread8(chip->pch_phub_extrom_base_address +
						PHUB_STATUS) != 0x00) {
		msleep(1);
		if (i == PHUB_TIMEOUT)
			return -ETIMEDOUT;
		i++;
	}

	iowrite32(PCH_PHUB_ROM_WRITE_DISABLE,
			chip->pch_phub_extrom_base_address + PHUB_CONTROL);

	return 0;
}

/**
 * pch_phub_read_serial_rom_val() - Read Serial ROM value
 * @chip:		Pointer to the PHUB register structure
 * @offset_address:	Serial ROM address offset value.
 * @data:		Serial ROM value to read.
 */
static void pch_phub_read_serial_rom_val(struct pch_phub_reg *chip,
					 unsigned int offset_address, u8 *data)
{
	unsigned int mem_addr;

	mem_addr = chip->pch_mac_start_address +
			pch_phub_mac_offset[offset_address];

	pch_phub_read_serial_rom(chip, mem_addr, data);
}

/**
 * pch_phub_write_serial_rom_val() - writing Serial ROM value
 * @chip:		Pointer to the PHUB register structure
 * @offset_address:	Serial ROM address offset value.
 * @data:		Serial ROM value.
 */
static int pch_phub_write_serial_rom_val(struct pch_phub_reg *chip,
					 unsigned int offset_address, u8 data)
{
	int retval;
	unsigned int mem_addr;

	mem_addr = chip->pch_mac_start_address +
			pch_phub_mac_offset[offset_address];

	r